/***************************************************************
 *                   Materials Object Library                  *
 *    Class Schmid_tensor : declaration for Schmid's tensor    *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.2.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2008 COLLARD Christophe
 * copyright © 2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::Schmid_tensor
    \brief Computes Schmid's tensor

    \htmlonly
    <FONT color="#838383">

    Schmid_tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Computes the symmetric Schmid tensor associated with \f$ p^\text{th} \f$ slip system: \n
    \f$ R_{ij}^p = (n^p_i m^p_j + n^p_j m^p_i) \f$ \n
    where \f$ \mathbf{n}^p \f$ is the unit vector normal to the slip plane and \f$ \mathbf{m}^p \f$ is the unit vector in the slip direction.

    \author copyright \htmlonly &#169; \endhtmlonly 2008 Christophe COLLARD \n
            copyright \htmlonly &#169; 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
    \version 2.2.0
    \date 2008
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type Schmid_tensor
#endif

#ifndef _Schmid_tensor_h
#define _Schmid_tensor_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATHS_H)
#include <math.h>
#endif

#if !defined(__STRING_H)
#include <string.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__SYMTENSORS2_H)
#include "MOL++/symtensors2.h"
#endif

using namespace std;

namespace materiol
{


//===========================================================
template <class T> class Schmid_tensor : public symtensor2<T>
//===========================================================
{
  public :
    Schmid_tensor () : symtensor2<T> () {} // default constructor
    Schmid_tensor (vector<T>, vector<T>);  // constructor with data given in argument
    Schmid_tensor (ifstream&);  // constructor with data stored in a file
    ~Schmid_tensor () {}        // destructor
};


//=====Private methods for Schmid tensor=======================================

//=====Public methods for slip_system==========================================


/*!
  \brief Computes the symmetric Schmid tensor associated with the \f$ p^\text{th} \f$ slip system

  \f$ \displaystyle R_{ij}^p = \frac{1}{2} (n^p_i m^p_j + n^p_j m^p_i) \f$, \n
  where \f$ \mathbf{n}^p \f$ is the unit vector normal to the habit plane and \f$ \mathbf{m}^p \f$ is the unit vector in the slip direction.
  \param n unit vector normal to the habit plane
  \param m unit vector in the slip direction
*/

//---------------------------------------------------------------------------------------------------------
template <class T> Schmid_tensor<T>::Schmid_tensor (vector<T> n, vector<T> m) : symtensor2<T> (3, 3, false)
//---------------------------------------------------------------------------------------------------------
{
  assert (n.dim()==3 && m.dim()==3);

  T n_norm = n.norm();
  T m_norm = m.norm();

  assert (n_norm > epsilon && m_norm > epsilon);

  n /= (T) n_norm;
  m /= (T) m_norm;

  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      (*this)(i,j)  = 0.5 * (n[i] * m[j] + n[j] * m[i]);
}


/*!
  \brief Computes the symmetric Schmid tensor associated with the \f$ p^\text{th} \f$ slip system

  \f$ \displaystyle R_{ij}^p = \frac{1}{2} (n^p_i m^p_j + n^p_j m^p_i) \f$, \n
  where \f$ \mathbf{n}^p \f$ is the unit vector normal to the habit plane and \f$ \mathbf{m}^p \f$ is the unit vector in the slip direction.
  \param file file where \f$ \mathbf{n}^p \f$ and \f$ \mathbf{m}^p \f$ are stored
*/

//-----------------------------------------------------------------------------------------------
template <class T> Schmid_tensor<T>::Schmid_tensor (ifstream& file) : symtensor2<T> (3, 3, false)
//-----------------------------------------------------------------------------------------------
{
  vector<T> n(3);  // allocates memory without initializing the vector to 0
  vector<T> m(3);  // same remark

  file >> n >> m;
  n = n/ (T) n.norm();
  m = m/ (T) m.norm();
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      (*this)(i,j)  = 0.5 * (n[i] * m[j] + n[j] * m[i]);
}


}


#endif
